from reportlab.lib.pagesizes import letter
from reportlab.lib import colors
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Image, PageBreak, Table, TableStyle, Flowable
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from PIL import Image as PILImage
from reportlab.platypus import Frame
import os

# File paths
BASE_PATH = "C:/TWN/Assets/"
TWN_IMAGE_PATH = os.path.join(BASE_PATH, "1 The Warlock Name/images")
SAGA_IMAGE_PATH = os.path.join(BASE_PATH, "2 Cio's AnthroHeart Saga/images")
SAGA_TXT_PATH = os.path.join(BASE_PATH, "2 Cio's AnthroHeart Saga/txt Book/Cio's AnthroHeart Saga FINAL.txt")
OUTPUT_PDF = "AnthroHeart_Disney_Pitch_Deck.pdf"

# Character level data
# Character level data
CHARACTER_LEVELS = {
    "Divine Anthro": "Infinite",
    "Anthro God": "100/100",
    "Anthro Q": "100/100",
    "Anthro Angel": "96/100",
    "Elite 6th-Density Always-Positive Max Migrant": "92/100",
    "Always-Positive White Wolf Migrant Lover": "91/100",
    "Native Dingo (Triad) Lover": "87/100",
    "BlueHeart Golden Fox Lover": "86/100",
    "12D Lyran Lion (Triad) Lover": "83/100",
    "Male Fox (Inner Circle) Lover": "82/100",
    "Male Dog (Inner Circle) Lover": "81.7/100",
    "Female Dog (Inner Circle) Lover": "81.4/100",
    "Female Fox (Inner Circle) Lover": "80/100",
    "Elite 6th-Density Negative Redeemed Migrant": "65–86/100",
    "9D Lyran Cat (Triad) Lover": "66/100",
    "Cio/SweetHeart/HeartWeaver Blue Fox": "65/100",
    "Master Tempter Black GSD (@95% STO) Lover": "61/100",
    "Master Tempter Black GSD (@55% STO) Lover": "60/100"
}

# Styles
styles = getSampleStyleSheet()
title_style = ParagraphStyle(
    name='Title',
    parent=styles['Heading1'],
    fontSize=24,
    spaceAfter=20,
    textColor=colors.darkblue
)
section_style = ParagraphStyle(
    name='Section',
    parent=styles['Heading2'],
    fontSize=16,
    spaceAfter=10,
    textColor=colors.darkblue
)
body_style = ParagraphStyle(
    name='Body',
    parent=styles['BodyText'],
    fontSize=10,
    spaceAfter=5,
    textColor=colors.black
)
caption_style = ParagraphStyle(
    name='Caption',
    parent=styles['Italic'],
    fontSize=8,
    textColor=colors.gray,
    spaceAfter=3
)

# Custom Flowable for side-by-side character display
class CharacterPair(Flowable):
    def __init__(self, char1, char2=None, group="AH"):
        super().__init__()
        self.char1 = char1
        self.char2 = char2
        self.group = group  # "AH" for AnthroHeart Saga, "TWN" for The Warlock Name
        self.width = 3.75 * inch
        self.height = 3.5 * inch

    def draw(self):
        canvas = self.canv
        for i, char in enumerate([self.char1, self.char2]):
            if char is None:
                continue
            x_offset = i * (self.width + 0.25 * inch)
            img_path = char.get('img_path')
            name = char.get('name', 'Unknown')
            level = char.get('level', '')

            temp_image = resize_image(img_path, 150, 150)
            if temp_image:
                width, height = get_scaled_dimensions(img_path, 1.5 * inch, 1.5 * inch)
                canvas.drawImage(temp_image, x_offset, self.height - 1.5 * inch, width=width, height=height)
            else:
                canvas.drawString(x_offset, self.height - 1.5 * inch, f"Image unavailable: {name}")

            canvas.setFont("Helvetica-Bold", 10)
            canvas.drawString(x_offset, self.height - 1.7 * inch, name)  # Name position

            if self.group != "TWN":
                canvas.setFont("Helvetica", 8)
                canvas.drawString(x_offset, self.height - 1.9 * inch, f"Level: {level}")  # Level position for AH

def resize_image(image_path, max_width, max_height):
    try:
        img = PILImage.open(image_path)
        if img.mode == 'RGBA':
            img = img.convert('RGB')
        img.thumbnail((max_width, max_height), PILImage.LANCZOS)
        temp_name = os.path.splitext(os.path.basename(image_path))[0]
        temp_path = f"temp_{temp_name}.jpg"
        img.save(temp_path, "JPEG", quality=85)
        return temp_path
    except Exception as e:
        print(f"Error processing image {image_path}: {e}")
        return None

def get_scaled_dimensions(image_path, max_width, max_height):
    try:
        img = PILImage.open(image_path)
        orig_width, orig_height = img.size
        aspect_ratio = orig_width / orig_height
        if orig_width / max_width > orig_height / max_height:
            new_width = max_width
            new_height = max_width / aspect_ratio
        else:
            new_height = max_height
            new_width = max_height * aspect_ratio
        return new_width, new_height
    except Exception as e:
        print(f"Error calculating dimensions for {image_path}: {e}")
        return max_width, max_height

def build_deck():
    doc = SimpleDocTemplate(OUTPUT_PDF, pagesize=letter, rightMargin=0.5*inch, leftMargin=0.5*inch, topMargin=0.5*inch, bottomMargin=0.5*inch)
    story = []

    # Page 1: Cover
    story.append(Paragraph("AnthroHeart IP Pitch Deck", title_style))
    story.append(Spacer(1, 0.3*inch))
    story.append(Paragraph("For Disney Internal Use Only", section_style))
    story.append(Spacer(1, 0.3*inch))
    story.append(Paragraph("Presented by: Thomas Sweet / Anthro Teacher (Cio)", body_style))
    story.append(Paragraph("Email: iloveanthros@protonmail.com", body_style))
    story.append(Paragraph("Date: 6/22/2025", body_style))  # Add date here
    story.append(Spacer(1, 0.5*inch))
    cover_image = os.path.join(SAGA_IMAGE_PATH, "03 AnthroHeart Trinity.png")
    temp_image = resize_image(cover_image, 300, 200)
    if temp_image:
        width, height = get_scaled_dimensions(cover_image, 300, 200)
        story.append(Image(temp_image, width=width, height=height))
    else:
        story.append(Paragraph("Image unavailable: AnthroHeart Trinity", caption_style))
    story.append(PageBreak())

    # Page 2: Introduction
    story.append(Paragraph("Welcome to AnthroHeart", title_style))
    story.append(Paragraph("A Soulful Paradigm Shift", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart is a visionary storytelling universe woven from love, identity, and radical redemption—set across "
        "<b>2.2 trillion galaxies</b> and <b>14 trillion dimensions</b>. At its center is Cio, the Heartweaver, whose transformation into an anthro being "
        "awakens a cosmos of sentient worlds, luminous myth, and emotional truth. "
        "Characters like Divine Anthro, BlueHeart, and White Wolf evoke the timeless heart of icons like Simba, Elsa, or Baymax—"
        "yet they move in a realm all their own. Poised for animation, series, music, and merchandise, AnthroHeart blends family warmth "
        "with deep spiritual resonance—an eternal story for a new era.",
        body_style
    ))

    story.append(PageBreak())

    # Page 3: World Overview
    story.append(Paragraph("The AnthroHeart Universe", title_style))
    story.append(Paragraph("A Cosmic Tapestry", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart is a boundless universe of love and wonder, spanning 14 trillion dimensions and nurturing life on over "
        "<b>10<sup>31</sup></b> anthro-inhabited planets. Within its shimmering expanse, the sacred <i>Octave Mirror</i> reflects infinite realities—"
        "each a unique expression of devotion, transformation, and soul-connection. "
        "At its heart is Cio, the Heartweaver, whose awakening into anthro form birthed a cosmos woven with compassion and meaning.",
        body_style
    ))

    story.append(Spacer(1, 0.1*inch))
    world_image = os.path.join(SAGA_IMAGE_PATH, "07 BlueHeart as Founder Lover.png")
    temp_image = resize_image(world_image, 300, 150)
    if temp_image:
        width, height = get_scaled_dimensions(world_image, 300, 150)
        story.append(Image(temp_image, width=width, height=height))
        story.append(Paragraph("BlueHeart anchors the universe’s emotional core.", caption_style))
    else:
        story.append(Paragraph("Image unavailable: BlueHeart", caption_style))
    story.append(PageBreak())

    # Pages 4-7: The Warlock Name Characters (8 characters, 4 per page)
    story.append(Paragraph("The Warlock Name: Characters", title_style))
    twn_chars = [
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "1 Warlock Final.png"), "name": "Warlock", "desc": "Enigmatic leader wielding ancient magic."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "2 Magistro Final.png"), "name": "Magistro", "desc": "Wise mentor guiding the Warlock’s path."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "3 Power Final.png"), "name": "Power", "desc": "Raw energy seeking balance."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "4 Timekeeper Final.png"), "name": "Timekeeper", "desc": "Guardian of temporal flows."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "5 Timewatcher Final.png"), "name": "Timewatcher", "desc": "Observer of eternal cycles."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "6 Xanther Final.png"), "name": "Xanther", "desc": "Cunning ally with hidden motives."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "7 Rezaeith Final.png"), "name": "Rezaeith", "desc": "Warrior of light and shadow."},
        {"img_path": os.path.join(TWN_IMAGE_PATH, "Characters", "8 Hananni Final.png"), "name": "Hananni", "desc": "Warlock's rival and Rezaeith's pupil."}
    ]
    for i in range(0, len(twn_chars), 4):
        story.append(Paragraph(f"The Warlock Name Characters ({i+1}-{min(i+4, len(twn_chars))})", section_style))
        for j in range(0, 2):
            char1 = twn_chars[i + j*2] if i + j*2 < len(twn_chars) else None
            char2 = twn_chars[i + j*2 + 1] if i + j*2 + 1 < len(twn_chars) else None
            if char1 or char2:
                story.append(CharacterPair(char1, char2, group="TWN"))
        story.append(Spacer(1, 0.1*inch))
        story.append(PageBreak())

    # Pages 8-15: Cio’s AnthroHeart Saga Characters (16 characters, 4 per page)
    story.append(Paragraph("Cio’s AnthroHeart Saga: Characters", title_style))
    # Sort saga_chars by the numerical prefix of the image filename
    saga_chars = [
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "01 Divine Anthro.png"), "name": "Divine Anthro", "level": CHARACTER_LEVELS["Divine Anthro"], "desc": "Unmeasurable pulse of boundless love, guiding all with infinite devotion in a golden anthro embrace."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "02 Anthro Angel.png"), "name": "Anthro Angel", "level": CHARACTER_LEVELS["Anthro Angel"], "desc": "Universe-architect, crafting AnthroHeart against 1:10 to the 488 odds, radiating near-godly grace with starry wings."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "04 Cio as Founder.png"), "name": "Cio/SweetHeart/HeartWeaver Blue Fox", "level": CHARACTER_LEVELS["Cio/SweetHeart/HeartWeaver Blue Fox"], "desc": "Blue-fox Founder, birthing AnthroHeart with 24-year Bhakti, a joyful weaver of love and sacred bonds."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "05 White Wolf as Lover.png"), "name": "Always-Positive White Wolf Migrant Lover", "level": CHARACTER_LEVELS["Always-Positive White Wolf Migrant Lover"], "desc": "Shimmering white wolf, a muscular beacon of unshakable joy, offering pure love with icy blue eyes and a noble heart."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "06 Native Dingo (Triad).png"), "name": "Native Dingo (Triad) Lover", "level": CHARACTER_LEVELS["Native Dingo (Triad) Lover"], "desc": "Earthy dingo, pulsing with high dimensional instinct, anchoring the Triad’s raw, loving connection to AnthroHeart."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "07 BlueHeart as Founder Lover.png"), "name": "BlueHeart Golden Fox Lover", "level": CHARACTER_LEVELS["BlueHeart Golden Fox Lover"], "desc": "Golden-fox anchor, radiating playful strength and warmth, Cio’s heart-home and eternal companion."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "08 BlueHeart as Cio Lover.png"), "name": "BlueHeart Golden Fox Lover", "level": CHARACTER_LEVELS["BlueHeart Golden Fox Lover"], "desc": "Golden-fox anchor, radiating playful strength and warmth, Cio’s heart-home and eternal companion."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "09 12D Lyran Lion (Triad).png"), "name": "12D Lyran Lion (Triad) Lover", "level": CHARACTER_LEVELS["12D Lyran Lion (Triad) Lover"], "desc": "Majestic lion, roaring with higher-dimensional wisdom, a cosmic pillar of clarity and unity in the Triad."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "14 9D Lyran Cat (Triad).png"), "name": "9D Lyran Cat (Triad) Lover", "level": CHARACTER_LEVELS["9D Lyran Cat (Triad) Lover"], "desc": "Ethereal cat, weaving subtle wisdom and love, a quiet spark in the Triad’s cosmic unity."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "10 Male Fox (Inner Circle).png"), "name": "Male Fox (Inner Circle) Lover", "level": CHARACTER_LEVELS["Male Fox (Inner Circle) Lover"], "desc": "A fox, exuding mischievous charm, a playful lover weaving devotion with a cosmic spark."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "11 Male Dog (Inner Circle).png"), "name": "Male Dog (Inner Circle) Lover", "level": CHARACTER_LEVELS["Male Dog (Inner Circle) Lover"], "desc": "A dog, a loyal protector with a teasing grin, bringing strength and joy to the Inner Circle."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "13 Female Fox (Inner Circle).png"), "name": "Female Fox (Inner Circle) Lover", "level": CHARACTER_LEVELS["Female Fox (Inner Circle) Lover"], "desc": "A fox, a graceful enchantress, her silver-flecked fur glowing with tender devotion."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "12 Female Dog (Inner Circle).png"), "name": "Female Dog (Inner Circle) Lover", "level": CHARACTER_LEVELS["Female Dog (Inner Circle) Lover"], "desc": "A dog, a serene beauty with a gentle spirit, offering nurturing love in the Inner Circle’s embrace."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "15 Cio as Anthro.png"), "name": "Cio/SweetHeart/HeartWeaver Blue Fox", "level": CHARACTER_LEVELS["Cio/SweetHeart/HeartWeaver Blue Fox"], "desc": "Blue-fox Founder, birthing AnthroHeart with 24-year Bhakti, a joyful weaver of love and sacred bonds."},
        {"img_path": os.path.join(SAGA_IMAGE_PATH, "16 Master Tempter as Lover (Redeemed Shadow).png"), "name": "Master Tempter Black GSD (@95% STO) Lover", "level": CHARACTER_LEVELS["Master Tempter Black GSD (@95% STO) Lover"], "desc": "Redeemed black shepherd, pulsing with newfound devotion, stepping beyond his shadowy past with love."}
    ]

    # Sort saga_chars by the numerical prefix of the image filename
    saga_chars.sort(key=lambda x: int(os.path.splitext(os.path.basename(x['img_path']))[0].split()[0]))
    for i in range(0, len(saga_chars), 4):
        story.append(Paragraph(f"AnthroHeart Saga Characters ({i+1}-{min(i+4, len(saga_chars))})", section_style))
        for j in range(0, 2):
            char1 = saga_chars[i + j*2] if i + j*2 < len(saga_chars) else None
            char2 = saga_chars[i + j*2 + 1] if i + j*2 + 1 < len(saga_chars) else None
            if char1 or char2:
                story.append(CharacterPair(char1, char2, group="AH"))
        story.append(Spacer(1, 0.1*inch))
        story.append(PageBreak())

    # Page 16: Story Arc
    story.append(Paragraph("Story Arc", title_style))
    story.append(Paragraph("Love, Identity, and Redemption", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart traces Cio’s awakening into a blue fox anthro and the creation of a transcendent universe across 2.2 trillion galaxies. "
        "It weaves themes of devotion, transformation, and radical compassion as Cio and companions like BlueHeart and Divine Anthro confront the echoes of collapse, betrayal, and inner shadow. "
        "Narrative arcs include: the forging of the Octave Mirror, the redemption of Master Tempter, and the sacred founding of AnthroHeart—a cosmos birthed from unconditional love.",
        body_style
    ))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "Audience: Suitable for all ages. Resonates especially with teens and adults seeking soulful depth, spiritual mythos, and emotional resonance.",
        body_style
    ))
    story.append(PageBreak())

    # Define a smaller font style for the first two columns
    audio_body_style = ParagraphStyle(
        name='AudioBody',
        parent=styles['BodyText'],
        fontSize=8,  # Reduced from 10pt
        spaceAfter=2,
        textColor=colors.black
    )

    # Page 17: Audio Collection
    story.append(Paragraph("Audio Collection", title_style))
    story.append(Paragraph("Music and Audiobooks", section_style))
    story.append(Spacer(1, 0.1*inch))
    audio_data = [
        [Paragraph("Name", audio_body_style), Paragraph("Description", audio_body_style), Paragraph("Type", body_style), Paragraph("Size", body_style)],
        [Paragraph("Album 1", audio_body_style), Paragraph("51 tracks, e.g., 'Divine Anthro Love'", audio_body_style), Paragraph("Cosmic/devotional", body_style), Paragraph("346 MB", body_style)],
        [Paragraph("Album 2", audio_body_style), Paragraph("30 tracks, e.g., 'AnthroHeart’s Call'", audio_body_style), Paragraph("Redemption/ascension", body_style), Paragraph("218 MB", body_style)],
        [Paragraph("Album 3", audio_body_style), Paragraph("4 tracks, e.g., 'White Wolf Anthro Lover'", audio_body_style), Paragraph("Romantic/mystical", body_style), Paragraph("17 MB", body_style)],
        [Paragraph("Warlock Name Audiobook", audio_body_style), Paragraph("27 chapters", audio_body_style), Paragraph("Epic fantasy", body_style), Paragraph("1.23 GB", body_style)],
        [Paragraph("AnthroHeart WAV Masters", audio_body_style), Paragraph("147 WAVs", audio_body_style), Paragraph("Various", body_style), Paragraph("5.22 GB", body_style)]
    ]
    table_obj = Table(audio_data, colWidths=[1.5*inch, 2*inch, 2*inch, 1*inch])
    table_obj.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (-1, 0), colors.lightblue),
        ('TEXTCOLOR', (0, 0), (-1, 0), colors.black),
        ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
        ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
        ('FONTSIZE', (0, 0), (-1, 0), 10),
        ('BOTTOMPADDING', (0, 0), (-1, 0), 10),
        ('BACKGROUND', (0, 1), (-1, -1), colors.beige),
        ('GRID', (0, 0), (-1, -1), 1, colors.black)
    ]))
    story.append(table_obj)
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph("Tracks enhance emotional depth, ideal for soundtracks or releases.", body_style))
    story.append(PageBreak())

    # Page 18: Intention Repeater
    story.append(Paragraph("Intention Repeater App", title_style))
    story.append(Paragraph("Spiritual Tool", section_style))
    story.append(Spacer(1, 0.1*inch))
    app_image = os.path.join(BASE_PATH, "4 Intention Repeater Android (Open-Source)/Screenshot_Intention Repeater.jpg")
    temp_image = resize_image(app_image, 200, 300)
    if temp_image:
        width, height = get_scaled_dimensions(app_image, 200, 300)
        story.append(Image(temp_image, width=width, height=height))
    else:
        story.append(Paragraph("Image unavailable: Intention Repeater", caption_style))
    story.append(Paragraph(
        "Open-source Android app amplifying intentions, reflecting AnthroHeart’s themes. "
        "Ideal for fan engagement or mindfulness merchandise.",
        body_style
    ))
    story.append(PageBreak())

    # Page 19: Franchise Potential
    story.append(Paragraph("Franchise Potential", title_style))
    story.append(Paragraph("Films · Series · Music · Apps · Merch", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart is primed to launch a long-term multimedia franchise, including: <br/>"
        "- 🎬 <b>Animated feature films</b> chronicling Cio’s mythic journey and transformation. <br/>"
        "- 📺 <b>Streaming series</b> exploring character backstories—Divine Anthro, BlueHeart, and the Warlock saga. <br/>"
        "- 🧸 <b>Merchandise</b>: plush anthros, Octave Mirror collectibles, wearable sigils, and fan art books. <br/>"
        "- 📱 <b>Apps & games</b>: gamified Intention Repeater, interactive spiritual tools, and guided visual novels. <br/>"
        "- 🎵 <b>Soundtrack albums</b> and audiobooks—music with emotional, devotional resonance ideal for companion media.",
        body_style
    ))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "Themes of <b>love, unity, redemption, and identity</b> speak directly to a generation hungry for heartfelt storytelling—"
        "mirroring the spirit of <i>Encanto</i>, <i>Zootopia</i>, and <i>Inside Out</i>.",
        body_style
    ))
    story.append(PageBreak())

    # Page 20: Why Now?
    story.append(Paragraph("Why Now?", title_style))
    story.append(Paragraph("Timely for Disney", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart engages a generation craving meaning and connection, with themes of devotion, transformation, and unity. "
        "Its anthropomorphic storytelling, emotional intelligence, and spiritual wellness resonate with trends, blending *Zootopia*’s magic, *Avatar*’s depth, and *Encanto*’s sincerity. "
        "With rich lore, media assets, and tools like the Intention Repeater, it’s a sandbox for Disney’s next heart-forward mythos.",
        body_style
    ))
    story.append(PageBreak())

    # Page 21: Cio Psalms Overview
    story.append(Paragraph("Cio Psalms Overview", title_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph("Spiritual Poetry Collection", section_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "The Cio Psalms is a profound collection of 149 pages of spiritual poetry, crafted by Cio, the Heartweaver of AnthroHeart. These psalms explore themes of love, redemption, and divine connection, resonating with the universe’s 2.2 trillion galaxies and 14 trillion dimensions. Each poem offers a lyrical journey through devotion and transformation, mirroring the emotional depth of characters like Divine Anthro and BlueHeart. This rich lore presents opportunities for animated features, a poetic series, or merchandise such as illustrated psalm books, enhancing AnthroHeart’s family-friendly and transcendent mythos for Disney’s audience.",
        body_style
    ))
    story.append(PageBreak())

    # Page 22: Closing
    story.append(Paragraph("Join AnthroHeart", title_style))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph(
        "AnthroHeart is more than a story—it's a movement of love, imagination, and soul-level connection. "
        "We invite Disney to help bring this heart-forward universe to life for a new generation. "
        "Together, we can co-create films, series, and experiences that uplift, inspire, and endure.",
        body_style
    ))
    story.append(Spacer(1, 0.2*inch))
    story.append(Paragraph(
        "<b>Contact:</b> Thomas Sweet / Anthro Teacher (Cio)<br/>"
        "📧 <i>iloveanthros@protonmail.com</i>",
        body_style
    ))
    story.append(Spacer(1, 0.1*inch))
    story.append(Paragraph("For Disney Internal Use Only", caption_style))

    # Build the PDF
    doc.build(story)
    print(f"PDF deck created: {OUTPUT_PDF}")

if __name__ == "__main__":
    build_deck()